/*
 * This program is free software; you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License, version 2.1 as published by the Free Software
 * Foundation.
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * program; if not, you can obtain a copy at http://www.gnu.org/licenses/old-licenses/lgpl-2.1.html
 * or from the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 *
 * Copyright (c) 2006 - 2009 Pentaho Corporation and Contributors.  All rights reserved.
 */

package org.pentaho.reporting.libraries.formula.function.math;

import java.math.BigDecimal;

import org.pentaho.reporting.libraries.formula.EvaluationException;
import org.pentaho.reporting.libraries.formula.FormulaContext;
import org.pentaho.reporting.libraries.formula.LibFormulaErrorValue;
import org.pentaho.reporting.libraries.formula.function.Function;
import org.pentaho.reporting.libraries.formula.function.ParameterCallback;
import org.pentaho.reporting.libraries.formula.lvalues.TypeValuePair;
import org.pentaho.reporting.libraries.formula.typing.coretypes.NumberType;
import org.pentaho.reporting.libraries.formula.util.NumberUtil;

/**
 * This function returns the average of the number sequence.
 *
 * @author Cedric Pronzato
 */
public class AverageFunction implements Function
{
  private static final long serialVersionUID = -5057715506050635450L;
  private SumFunction sumFunction;

  public AverageFunction()
  {
    sumFunction = new SumFunction();
  }

  protected AverageFunction(final SumFunction sumFunction)
  {
    this.sumFunction = sumFunction;
  }

  public String getCanonicalName()
  {
    return "AVERAGE";
  }

  public TypeValuePair evaluate(final FormulaContext context,
                                final ParameterCallback parameters)
      throws EvaluationException
  {
    final TypeValuePair sum = sumFunction.evaluate(context, parameters);

    final Number n = context.getTypeRegistry().convertToNumber(sum.getType(), sum.getValue());
    if (n == null)
    {
      throw EvaluationException.getInstance(LibFormulaErrorValue.ERROR_INVALID_ARGUMENT_VALUE);
    }
    final BigDecimal divident = NumberUtil.getAsBigDecimal(n);
    final BigDecimal divisor = new BigDecimal(parameters.getParameterCount());
    final BigDecimal avg = NumberUtil.divide(divident, divisor);
    return new TypeValuePair(NumberType.GENERIC_NUMBER, avg);
  }
}
