import os
import tempfile
import unittest

from cvise.passes.abstract import PassResult
from cvise.passes.ternary import TernaryPass


class TernaryBTestCase(unittest.TestCase):
    def setUp(self):
        self.pass_ = TernaryPass('b')

    def test_b(self):
        with tempfile.NamedTemporaryFile(mode='w', delete=False) as tmp_file:
            tmp_file.write('int res = a ? b : c;\n')

        state = self.pass_.new(tmp_file.name)
        (_, state) = self.pass_.transform(tmp_file.name, state, None)

        with open(tmp_file.name) as variant_file:
            variant = variant_file.read()

        os.unlink(tmp_file.name)

        self.assertEqual(variant, 'int res = b;\n')

    def test_parens(self):
        with tempfile.NamedTemporaryFile(mode='w', delete=False) as tmp_file:
            tmp_file.write('int res = (a != 0) ? (b + 5) : c;\n')

        state = self.pass_.new(tmp_file.name)
        (_, state) = self.pass_.transform(tmp_file.name, state, None)

        with open(tmp_file.name) as variant_file:
            variant = variant_file.read()

        os.unlink(tmp_file.name)

        self.assertEqual(variant, 'int res = (b + 5);\n')

    def test_all_b(self):
        with tempfile.NamedTemporaryFile(mode='w', delete=False) as tmp_file:
            tmp_file.write('// no ? match :\nint res = a ? (ba ? bb : bc) : c\nint sec = t ? u : v\n')

        state = self.pass_.new(tmp_file.name)
        (result, state) = self.pass_.transform(tmp_file.name, state, None)

        while result == PassResult.OK:
            state = self.pass_.advance_on_success(tmp_file.name, state)
            (result, state) = self.pass_.transform(tmp_file.name, state, None)

        with open(tmp_file.name) as variant_file:
            variant = variant_file.read()

        os.unlink(tmp_file.name)

        self.assertEqual(variant, '// match res = (bb)\nint sec = u\n')

    def test_all_b_2(self):
        with tempfile.NamedTemporaryFile(mode='w', delete=False) as tmp_file:
            tmp_file.write('// no ? match :!\nint res = a ? (ba ? bb : bc) : c\nint sec = t ? u : v\n')

        state = self.pass_.new(tmp_file.name)
        (result, state) = self.pass_.transform(tmp_file.name, state, None)

        iteration = 0

        while result == PassResult.OK and iteration < 5:
            state = self.pass_.advance_on_success(tmp_file.name, state)
            (result, state) = self.pass_.transform(tmp_file.name, state, None)
            iteration += 1

        with open(tmp_file.name) as variant_file:
            variant = variant_file.read()

        os.unlink(tmp_file.name)

        self.assertEqual(iteration, 3)
        self.assertEqual(variant, '// no ? match :!\nint res = (bb)\nint sec = u\n')

    def test_no_success(self):
        with tempfile.NamedTemporaryFile(mode='w', delete=False) as tmp_file:
            tmp_file.write('// no ? match :\nint res = a ? (ba ? bb : bc) : c\nint sec = t ? u : v\n')

        state = self.pass_.new(tmp_file.name)
        (result, state) = self.pass_.transform(tmp_file.name, state, None)

        iteration = 0

        while result == PassResult.OK and iteration < 6:
            with tempfile.NamedTemporaryFile(mode='w', delete=False) as tmp_file:
                tmp_file.write('// no ? match :\nint res = a ? (ba ? bb : bc) : c\nint sec = t ? u : v\n')

            state = self.pass_.advance(tmp_file.name, state)
            (result, state) = self.pass_.transform(tmp_file.name, state, None)
            iteration += 1

        os.unlink(tmp_file.name)

        self.assertEqual(iteration, 4)


class TernaryCTestCase(unittest.TestCase):
    def setUp(self):
        self.pass_ = TernaryPass('c')

    def test_c(self):
        with tempfile.NamedTemporaryFile(mode='w', delete=False) as tmp_file:
            tmp_file.write('int res = a ? b : c;\n')

        state = self.pass_.new(tmp_file.name)
        (_, state) = self.pass_.transform(tmp_file.name, state, None)

        with open(tmp_file.name) as variant_file:
            variant = variant_file.read()

        os.unlink(tmp_file.name)

        self.assertEqual(variant, 'int res = c;\n')
